package com.ebaiyihui.family.doctor.server.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ebaiyihui.family.doctor.common.dto.NotifyConsultDataDTO;
import com.ebaiyihui.family.doctor.common.dto.SendImMsgDTO;
import com.ebaiyihui.family.doctor.common.dto.SignedDoctorDTO;
import com.ebaiyihui.family.doctor.server.common.constants.CommonConstants;
import com.ebaiyihui.family.doctor.server.common.enums.ImSignStatus;
import com.ebaiyihui.family.doctor.server.common.enums.SignStatus;
import com.ebaiyihui.family.doctor.server.common.enums.StatusEnum;
import com.ebaiyihui.family.doctor.server.entity.MobileBenefitPackageEntity;
import com.ebaiyihui.family.doctor.server.entity.PatientSignEntity;
import com.ebaiyihui.family.doctor.server.entity.ScheduleRecordEntity;
import com.ebaiyihui.family.doctor.server.exception.BusinessException;
import com.ebaiyihui.family.doctor.server.mapper.MobileBenefitPackageMapper;
import com.ebaiyihui.family.doctor.server.mapper.PatientSignMapper;
import com.ebaiyihui.family.doctor.server.mapper.ScheduleRecordMapper;
import com.ebaiyihui.family.doctor.server.rabbitmq.RabbitProduct;
import com.ebaiyihui.family.doctor.server.service.GoEasyPushService;
import com.ebaiyihui.family.doctor.server.service.ImChatTemplate;
import com.ebaiyihui.family.doctor.server.service.ImMsgTemplateService;
import com.ebaiyihui.family.doctor.server.service.PatientService;
import com.ebaiyihui.family.doctor.server.util.DateUtils;
import com.ebaiyihui.family.doctor.server.util.ThirdOrderPushUtil;
import com.ebaiyihui.family.doctor.server.util.UUIDUtil;
import com.ebaiyihui.family.doctor.server.vo.OrderTaskVo;
import com.ebaiyihui.framework.response.BaseResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;

/**
 * @ClassName: PatientServiceImpl
 * @Author:yanliang
 * @Date:2024/3/13 09:51
 * @Description
 */
@Service
@Slf4j
public class PatientServiceImpl implements PatientService {

    @Autowired
    private PatientSignMapper patientSignMapper;

    @Autowired
    private MobileBenefitPackageMapper mobileBenefitPackageMapper;

    @Autowired
    private ScheduleRecordMapper scheduleRecordMapper;

    @Autowired
    private ImMsgTemplateService imMsgTemplateService;

    @Autowired
    private GoEasyPushService goEasyPushService;

    @Autowired
    private ImChatTemplate imChatTemplate;

    @Autowired
    private ThirdOrderPushUtil thirdOrderPushUtil;

    @Autowired
    private RabbitProduct rabbitProduct;


    @Override
    public BaseResponse<String> signedDoctor(SignedDoctorDTO reqVo) {

        MobileBenefitPackageEntity entity = mobileBenefitPackageMapper.queryByPhone(reqVo.getPhone(), reqVo.getPackageOrderId());
        if (ObjectUtils.isEmpty(entity)) {
            return BaseResponse.error("没有查询到有关权益");
        }

        String admId = UUIDUtil.getUUID();
        Date curDate = new Date();
        Date signEndTime = DateUtils.getAfterDay(curDate, CommonConstants.DAY);
        Date signStartTime = new Date();
        try {
            if (!StringUtils.isEmpty(entity.getBenefitsEndTime())) {
                signEndTime = DateUtils.parseDate(entity.getBenefitsEndTime(), DateUtils.FULL_FORMAT);
                signStartTime = DateUtils.parseDate(entity.getBenefitsStartTime(), DateUtils.FULL_FORMAT);
            }
        } catch (Exception e) {
            return BaseResponse.error("没有查询到有关权益");
        }
        Long doctorId = reqVo.getDoctorId();
        String doctorName = reqVo.getDoctorName();
        Long deptId = reqVo.getDeptId();
        String deptName = reqVo.getDeptName();
        List<PatientSignEntity> patientSignEntities = new ArrayList<>();
        if (!SignStatus.NORMAL.getValue().equals(reqVo.getSignStatus())) {
            patientSignEntities = getPatientSignList(reqVo.getPhone(), reqVo.getPackageOrderId(),
                    reqVo.getSignStatus(), StatusEnum.IN_CONSULTATION.getValue());
            if (!patientSignEntities.isEmpty()) {
                return BaseResponse.error("已经有进行中的签约或者改签医生，不能再进行签约或者改签！！！");
            }
        } else {
            QueryWrapper<PatientSignEntity> psWrapper = new QueryWrapper<>();
            PatientSignEntity patientSign = new PatientSignEntity();
            patientSign.setPatientPhone(reqVo.getPhone());
            patientSign.setPackageOrderId(reqVo.getPackageOrderId());
            patientSign.setSignStatus(SignStatus.NORMAL.getValue());
            patientSign.setStatus(StatusEnum.IN_CONSULTATION.getValue());
            psWrapper.setEntity(patientSign);
            PatientSignEntity patientSignEntity = patientSignMapper.selectOne(psWrapper);
            if (Objects.nonNull(patientSignEntity)) {
                return BaseResponse.success(patientSignEntity.getAdmId());
            }

            // 查询排班医生信息
            ScheduleRecordEntity scheduleRecordEntity = getScheduleDoctor(reqVo.getPatientId());
            doctorId = Long.valueOf(scheduleRecordEntity.getDoctorId());
            doctorName = scheduleRecordEntity.getDoctorName();
            deptId = Long.valueOf(scheduleRecordEntity.getDeptId());
            deptName = scheduleRecordEntity.getDeptName();
        }

        // 改签或者排班医生获取签约的id
        if (!SignStatus.SIGNED.getValue().equals(reqVo.getSignStatus())) {
            patientSignEntities = getPatientSignList(reqVo.getPhone(), reqVo.getPackageOrderId(),
                    SignStatus.SIGNED.getValue(), StatusEnum.IN_CONSULTATION.getValue());
        }

        // 处理换绑医生
        if ((Objects.nonNull(reqVo.getAdmId()) || Objects.nonNull(reqVo.getSignSeqId())
                && SignStatus.CANCEL.getValue().equals(reqVo.getSignStatus()))) {
            QueryWrapper<PatientSignEntity> psWrapper = new QueryWrapper<>();
            PatientSignEntity patientSign = new PatientSignEntity();
            if (Objects.nonNull(reqVo.getAdmId())) {
                patientSign.setAdmId(reqVo.getAdmId());
            }
            if (Objects.nonNull(reqVo.getSignSeqId())) {
                patientSign.setId(Long.valueOf(reqVo.getSignSeqId()));
            }
            psWrapper.setEntity(patientSign);
            PatientSignEntity patientSignEntity = patientSignMapper.selectOne(psWrapper);
            if (Objects.nonNull(patientSignEntity)) {
                signEndTime = patientSignEntity.getSignEndTime();
                patientSignEntity.setStatus(StatusEnum.FINISH_APPLY.getValue());
                patientSignEntity.setSignStatus(SignStatus.CANCEL.getValue());
                patientSignEntity.setSubStatus(2);
                patientSignEntity.setSignEndTime(curDate);
                patientSignMapper.updateById(patientSignEntity);
            }
        }

        PatientSignEntity patientSignEntity = new PatientSignEntity();
        patientSignEntity.setAdmId(admId);
        patientSignEntity.setAppCode(CommonConstants.APP_CODE);
        patientSignEntity.setOrganId(CommonConstants.ORGAN_CODE);
        patientSignEntity.setOrganName(CommonConstants.ORGAN_NAME);
        patientSignEntity.setDoctorId(doctorId);
        patientSignEntity.setDoctorName(doctorName);
        patientSignEntity.setDeptId(deptId);
        patientSignEntity.setDeptName(deptName);
        patientSignEntity.setSignStatus(SignStatus.CANCEL.getValue().equals(reqVo.getSignStatus()) ? SignStatus.SIGNED.getValue() : reqVo.getSignStatus());
        patientSignEntity.setPatientPhone(reqVo.getPhone());
        patientSignEntity.setCredNo(reqVo.getCredNo());
        patientSignEntity.setPatientId(reqVo.getPatientId());
        patientSignEntity.setPatientName(reqVo.getName());
        patientSignEntity.setPatientUserId(entity.getUserId());
        patientSignEntity.setStatus(StatusEnum.IN_CONSULTATION.getValue());
        patientSignEntity.setPackageOrderId(reqVo.getPackageOrderId());
        patientSignEntity.setSignEndTime(signEndTime);
        if (!patientSignEntities.isEmpty()) {
            patientSignEntity.setSignId(patientSignEntities.get(0).getSignId());
        }

        patientSignMapper.insert(patientSignEntity);

        //创建IM
        BaseResponse createImResponse = imChatTemplate.createImSession(patientSignEntity);
        if (!createImResponse.isSuccess()) {
            log.info("创建IM失败:{}", createImResponse.getMsg());
            throw new BusinessException("创建IM失败");
        }

        SendImMsgDTO sendImMsgDTO = new SendImMsgDTO();

        // 推送im主动消息
        sendImMsgDTO.setAdmId(admId);
        sendImMsgDTO.setAppCode(CommonConstants.APP_CODE);
        sendImMsgDTO.setOrganId(CommonConstants.ORGAN_CODE);
        sendImMsgDTO.setType(SignStatus.SIGNED.getValue().equals(reqVo.getSignStatus()) ? ImSignStatus.SIGN_SUC.getValue() :
                SignStatus.CANCEL.getValue().equals(reqVo.getSignStatus()) ? ImSignStatus.REISSUE_SUC.getValue() :
                        ImSignStatus.INQUIRY_SCH_DOC.getValue());

        if (!SignStatus.NORMAL.getValue().equals(reqVo.getSignStatus())) {
            // 推送第三方权益使用
            NotifyConsultDataDTO notifyConsultDataDTO = new NotifyConsultDataDTO();
            notifyConsultDataDTO.setActivateOrderId(reqVo.getPackageOrderId());
            notifyConsultDataDTO.setConsultOrderNo(admId);
            notifyConsultDataDTO.setStatus(String.valueOf(reqVo.getSignStatus()));
            notifyConsultDataDTO.setProductId(entity.getProductId());
            notifyConsultDataDTO.setPatientId(entity.getUserId());
            notifyConsultDataDTO.setExpertName(reqVo.getDoctorName());
            notifyConsultDataDTO.setOrderDate(DateUtils.dateToFullString(patientSignEntity.getCreateTime()));
            notifyConsultDataDTO.setSupplierCode(CommonConstants.SUPPLIER_CODE_JTYS);
            thirdOrderPushUtil.syncBenefitUsedOrder(notifyConsultDataDTO);
        } else {
            QueryWrapper<PatientSignEntity> psWrapper = new QueryWrapper<>();
            PatientSignEntity patientSign = new PatientSignEntity();
            if (Objects.nonNull(reqVo.getPhone())) {
                patientSign.setPatientPhone(reqVo.getPhone());
            }
            psWrapper.setEntity(patientSign);
            if (Objects.nonNull(reqVo.getPhone())) {
                psWrapper.orderByDesc("createTime");
                psWrapper.last("limit 1");
            }
            PatientSignEntity pse = patientSignMapper.selectOne(psWrapper);
            if (null == pse) {
                throw new BusinessException("未查询到签约信息,不能获取排班医生！");
            }
            sendImMsgDTO.setOldAdmId(pse.getAdmId());
        }

        //放入MQ任务,处理订单到时结束
        if ("0".equals(entity.getIsRenew()) || SignStatus.NORMAL.getValue().equals(reqVo.getSignStatus())) {
            OrderTaskVo orderTaskVo = new OrderTaskVo();
            orderTaskVo.setType(1);
            orderTaskVo.setId(String.valueOf(patientSignEntity.getId()));
            if ("0".equals(entity.getIsRenew())) {
                orderTaskVo.setExpireTime(DateUtils.secondBetween(signStartTime, signEndTime));
            }
            rabbitProduct.sendDelay(orderTaskVo);
        }

        // 推送第三方签名相关订单信息
        thirdOrderPushUtil.pushSignedOrder(String.valueOf(reqVo.getDoctorId()), reqVo.getPhone(),
                SignStatus.SIGNED.getValue().equals(reqVo.getSignStatus()) ? String.valueOf(patientSignEntity.getId()) :
                        String.valueOf(patientSignEntity.getSignId()),
                reqVo.getPackageOrderId(), reqVo.getSignStatus());

        imMsgTemplateService.sendImMsg(sendImMsgDTO);

        // 推送app提示语
        goEasyPushService.newOrderToReceive(admId);

        return BaseResponse.success(admId);
    }

    @Override
    public BaseResponse<PatientSignEntity> getSignedDoctor(SignedDoctorDTO reqVo) {
        QueryWrapper<PatientSignEntity> psWrapper = new QueryWrapper<>();
        PatientSignEntity patientSign = new PatientSignEntity();
        if (Objects.nonNull(reqVo.getAdmId())) {
            patientSign.setAdmId(reqVo.getAdmId());
        }
        if (Objects.nonNull(reqVo.getPhone())) {
            patientSign.setPatientPhone(reqVo.getPhone());
        }
        psWrapper.setEntity(patientSign);
        if (Objects.nonNull(reqVo.getPhone())) {
            psWrapper.orderByDesc("create_time");
            psWrapper.last("limit 1");
        }

        PatientSignEntity patientSignEntity = patientSignMapper.selectOne(psWrapper);
        if (null == patientSignEntity) {
            throw new BusinessException("未查询到签约医生信息！");
        }
        return BaseResponse.success(patientSignEntity);
    }

    /**
     * 随访查询一个排班医生
     *
     * @param patientId
     * @return
     */
    public ScheduleRecordEntity getScheduleDoctor(String patientId) {
        String date = DateUtils.dateToSimpleString(new Date());
        String hourAndSecond = DateUtils.getHourAndSecond();
        QueryWrapper<PatientSignEntity> psWrapper = new QueryWrapper<>();
        PatientSignEntity patientSign = new PatientSignEntity();
        patientSign.setPatientId(patientId);
        patientSign.setSignStatus(SignStatus.NORMAL.getValue());
        patientSign.setStatus(StatusEnum.IN_CONSULTATION.getValue());
        psWrapper.setEntity(patientSign);
        List<PatientSignEntity> patientSignEntities = patientSignMapper.selectList(psWrapper);
        if (CollectionUtils.isEmpty(patientSignEntities)) {
            throw new BusinessException("已有进行中排班医生沟通,请勿重复点击");
        }
        // 查询排班医生信息
        List<ScheduleRecordEntity> scheduleRecordEntities = scheduleRecordMapper.selectByDate(CommonConstants.APP_CODE, 4, date, 1, hourAndSecond);
        log.info("当前排班医生:{}", JSON.toJSONString(scheduleRecordEntities));

        if (CollectionUtils.isEmpty(scheduleRecordEntities)) {
            throw new BusinessException("暂无排班医生,请稍后重试");
        }

        ScheduleRecordEntity scheduleRecordEntity = new ScheduleRecordEntity();

        Random random = new Random();
        int i = random.nextInt(scheduleRecordEntities.size() + 1);
        if (i == 0) {
            i = i + 1;
        }
        scheduleRecordEntity = scheduleRecordEntities.get(i - 1);

        log.info("当前下单的医生:{}", JSON.toJSONString(scheduleRecordEntity));
        return scheduleRecordEntity;
    }


    /**
     * 查询患者签名信息
     *
     * @param phone
     * @param packageOrderId
     * @param signStatus
     * @param status
     * @return
     */
    List<PatientSignEntity> getPatientSignList(String phone, String packageOrderId, Integer signStatus, Integer status) {
        QueryWrapper<PatientSignEntity> psWrapper = new QueryWrapper<>();
        PatientSignEntity patientSign = new PatientSignEntity();
        patientSign.setPatientPhone(phone);
        patientSign.setPackageOrderId(packageOrderId);
        patientSign.setSignStatus(signStatus);
        patientSign.setStatus(status);
        psWrapper.setEntity(patientSign);
        psWrapper.orderByDesc("create_time");
        List<PatientSignEntity> patientSignEntities = patientSignMapper.selectList(psWrapper);
        return patientSignEntities;
    }
}
