package com.ebaiyihui.family.doctor.server.util;

import com.alibaba.fastjson.JSON;
import com.ning.http.client.AsyncHttpClient;
import com.ning.http.client.AsyncHttpClientConfig;
import com.ning.http.client.Response;
import org.apache.http.client.ClientProtocolException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLConnection;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

/**
 * @ClassName: HttpKit
 * @Author:yanliang
 * @Date:2024/3/18 15:54
 * @Description
 */

public final class HttpKit {

    private static Logger log = LoggerFactory.getLogger(HttpKit.class);
    private static final String DEFAULT_CHARSET = "UTF-8";
    /**
     * 请求超时时间
     */
    private static final int SO_TIMEOUT = 60000;
    /**
     * 最大允许连接数
     */
    private static final int MAX_TOTAL_CONNECTION = 2000;
    private static final int MAX_PER_HOST = 2000;
    /**
     * 异步请求对象
     */
    private static AsyncHttpClient asyncHttpClient;

    /**
     * 默认请求类型
     */
    private static final String DEFAULT_CONTENT_TYPE = "application/json";
    private static final String CONTENT_TYPE = "Content-Type";

    /**
     * 设置异步请求参数
     */
    static {
        AsyncHttpClientConfig.Builder builder = new AsyncHttpClientConfig.Builder();
        builder.setMaxConnections(MAX_TOTAL_CONNECTION);
        builder.setMaxConnectionsPerHost(MAX_PER_HOST);
        builder.setRequestTimeout(SO_TIMEOUT);
        asyncHttpClient = new AsyncHttpClient(builder.build());
    }

    /**
     * @return 返回类型:
     * @throws IOException
     * @throws UnsupportedEncodingException
     * @throws NoSuchProviderException
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     * @description 功能描述: get 请求
     */
    public static String get(String url, Map<String, String> params,
                             Map<String, String> headers) throws IOException, ExecutionException,
            InterruptedException {
        log.info("url{}", url);
        AsyncHttpClient.BoundRequestBuilder builder = asyncHttpClient.prepareGet(url);
        builder.setBodyEncoding(DEFAULT_CHARSET);
        if (params != null && !params.isEmpty()) {
            Set<String> keys = params.keySet();
            for (String key : keys) {
                builder.addQueryParam(key, params.get(key));
            }
        }

        if (headers != null && !headers.isEmpty()) {
            Set<String> keys = headers.keySet();
            for (String key : keys) {
                builder.addHeader(key, params.get(key));
            }
        }
        log.info("开始请求，请求地址:{},请求json参数:{}", url);
        Future<Response> f = builder.execute();
        String body = f.get().getResponseBody(DEFAULT_CHARSET);
        log.info("请求结束,响应结果:{}", body);
        return body;
    }

    /**
     * @return 返回类型:
     * @throws IOException
     * @throws UnsupportedEncodingException
     * @throws NoSuchProviderException
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     * @description 功能描述: get 请求
     */
    public static String get(String url) throws KeyManagementException, NoSuchAlgorithmException,
            NoSuchProviderException, UnsupportedEncodingException,
            IOException, ExecutionException, InterruptedException {
        return get(url, null);
    }

    /**
     * @return 返回类型:
     * @throws IOException
     * @throws NoSuchProviderException
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     * @throws UnsupportedEncodingException
     * @description 功能描述: get 请求
     */
    public static String get(String url,
                             Map<String, String> params) throws KeyManagementException,
            NoSuchAlgorithmException,
            NoSuchProviderException,
            UnsupportedEncodingException, IOException,
            ExecutionException, InterruptedException {
        return get(url, params, null);
    }

    /**
     * @return 返回类型:
     * @throws IOException
     * @throws NoSuchProviderException
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     * @description 功能描述: POST 请求
     */
    public static String post(String url, Map<String, String> params) throws IOException,
            ExecutionException,
            InterruptedException {
        AsyncHttpClient.BoundRequestBuilder builder = asyncHttpClient.preparePost(url);
        builder.setBodyEncoding(DEFAULT_CHARSET);
        if (params != null && !params.isEmpty()) {
            Set<String> keys = params.keySet();
            for (String key : keys) {
                builder.addQueryParam(key, params.get(key));
            }
        }
        Future<Response> f = builder.execute();
        String body = f.get().getResponseBody(DEFAULT_CHARSET);
        return body;
    }

    public static String post(String url, String s) throws IOException, ExecutionException,
            InterruptedException {
        AsyncHttpClient.BoundRequestBuilder builder = asyncHttpClient.preparePost(url);
        builder.setBodyEncoding(DEFAULT_CHARSET);
        builder.setBody(s);
        Future<Response> f = builder.execute();
        String body = f.get().getResponseBody(DEFAULT_CHARSET);
        return body;
    }

    public static String jsonPost(String url, String bodyJsonStr) throws IOException,
            ExecutionException,
            InterruptedException {
        // 设置post的基础请求信息
        AsyncHttpClient.BoundRequestBuilder builder = asyncHttpClient.preparePost(url);
        builder.setHeader("Content-Type", "application/json");

        builder.setBodyEncoding(DEFAULT_CHARSET);
        builder.setBody(bodyJsonStr);
        log.info("开始请求，请求地址:{},请求json参数:{}", url, bodyJsonStr);
        Future<Response> f = builder.execute();
        String body = f.get().getResponseBody(DEFAULT_CHARSET);
        log.info("请求结束,响应结果:{}", body);
        return body;
    }

    public static String jsonPost(String url, String bodyJsonStr,
                                  Map<String, String> headerMap) throws IOException,
            ExecutionException,
            InterruptedException {
        if (headerMap == null || headerMap.size() == 0) {
            return jsonPost(url, bodyJsonStr);
        }
        // 设置post的基础请求信息
        AsyncHttpClient.BoundRequestBuilder builder = asyncHttpClient.preparePost(url);
        builder.setHeader("Content-Type", "application/json");
        builder.setBodyEncoding(DEFAULT_CHARSET);
        builder.setBody(bodyJsonStr);
        // 处理 header
        headerMap.forEach((key, value) -> builder.setHeader(key, value));
        log.info("builder:{}" + JSON.toJSONString(builder));
        log.info("开始请求，请求地址:{},请求json参数:{}", url, bodyJsonStr);
        Future<Response> f = builder.execute();
        String body = f.get().getResponseBody(DEFAULT_CHARSET);
        log.info("请求结束,响应结果:{}", body);
        return body;
    }

    public static String doGetHader(String url, String params, Map<String, Object> headers) throws ClientProtocolException, IOException {
        AsyncHttpClient.BoundRequestBuilder builder = asyncHttpClient.prepareGet(url);
        builder.setBodyEncoding(DEFAULT_CHARSET);
        builder.setBody(params);
        if (headers != null && !headers.isEmpty()) {
            headers.forEach((key, value) -> builder.setHeader(key, String.valueOf(value)));
        }
        builder.setHeader(CONTENT_TYPE, DEFAULT_CONTENT_TYPE);
        Future<Response> f = builder.execute();
        String body = null;
        try {
            body = f.get().getResponseBody(DEFAULT_CHARSET);
        } catch (Exception e) {
            log.error("doGetHader请求 发生异常,url ={},e={}", url, e);
        }
        return body;
    }

    /**
     * 向指定URL发送GET方法的请求
     *
     * @param url   发送请求的URL
     * @param param 请求参数，请求参数应该是 name1=value1&name2=value2 的形式。
     * @return URL 所代表远程资源的响应结果
     */
    public static String sendGet(String url, String param) {
        String result = "";
        BufferedReader in = null;
        try {
            String urlNameString = param == null ? url : (url + "?" + param);
            URL realUrl = new URL(urlNameString);
            // 打开和URL之间的连接
            URLConnection connection = realUrl.openConnection();
            // 设置通用的请求属性
            connection.setRequestProperty("accept", "*/*");
            connection.setRequestProperty("connection", "Keep-Alive");
            connection.setRequestProperty("user-agent",
                    "Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.17 (KHTML, like Gecko) Chrome/24.0.1312.56 Safari/537.17");
            // 建立实际的连接
            connection.connect();
            // 定义 BufferedReader输入流来读取URL的响应
            in = new BufferedReader(new InputStreamReader(connection.getInputStream()));
            String line;
            while ((line = in.readLine()) != null) {
                result += line;
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error("发送GET请求出现异常:" + e.getMessage(), e);
        }
        // 使用finally块来关闭输入流
        finally {
            try {
                if (in != null) {
                    in.close();
                }
            } catch (Exception e2) {
                e2.printStackTrace();
            }
        }
        return result;
    }

    public static String sendGet(String url) {
        return sendGet(url, null);
    }
}
