package com.ebaiyihui.family.doctor.server.util;

import com.alibaba.fastjson.JSONObject;
import com.ebaiyihui.family.doctor.common.bo.Result;
import com.ebaiyihui.family.doctor.common.dto.*;
import com.ebaiyihui.family.doctor.common.vo.AccessTokenVo;
import com.ebaiyihui.family.doctor.server.common.constants.CommonConstants;
import com.ebaiyihui.family.doctor.server.common.constants.ProjProperties;
import com.ebaiyihui.family.doctor.server.common.constants.UrlConstants;
import com.ebaiyihui.family.doctor.server.common.enums.SignTypeEnum;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * @ClassName: ThirdOrderPushUtil
 * @Author:yanliang
 * @Date:2024/3/18 14:49
 * @Description
 */
@Slf4j
@Component
public class ThirdOrderPushUtil {

    @Autowired
    private ProjProperties projProperties;


    public <T> Map<String, String> getThirdHead(T t) {
        log.info("业务入参：{}", t);
        String param = JSONObject.toJSONString(t);
        log.info("业务数据：{}", param);
        String assessToken = getAssessToken();
        Map<String, String> header = new HashMap<>();
        header.put("Authorization", assessToken);
        Map<String, String> map = null;
        try {
            map = SignUtils.paramToMap(t);
        } catch (Exception e) {
            log.error("参数转换失败：{}", e);
        }
        String sign = SignUtils.getSignatureToStr(map, CommonConstants.SIGN_SECRET);
        header.put("sign", sign);
        header.put("timestamp", String.valueOf(new Date().getTime()));
        return header;
    }

    public String getAssessToken() {
        Map<String, String> param = new HashMap<>(16);
        param.put("grantType", CommonConstants.GRANT_TYPE);
        param.put("clientId", projProperties.getFamilyDoctorThirdClientId());
        param.put("clientSecret", projProperties.getFamilyDoctorThirdClientSecret());
        try {
            String result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.GET_TOKEN_URL, JSONObject.toJSONString(param));
            Result<AccessTokenVo> resultResp = JSONObject.parseObject(result, Result.class);
            if (resultResp.isSuccess()) {
                AccessTokenVo accessTokenVo = JSONObject.parseObject(String.valueOf(resultResp.getData()), AccessTokenVo.class);
                return accessTokenVo.getTokenType() + " " + accessTokenVo.getAccessToken();
            }
        } catch (Exception e) {
            log.error("错误信息：{}", e);
        }
        return null;
    }

    /**
     * 推送签名信息
     *
     * @param doctorId
     * @param phone
     * @param signId
     * @param packageOrderId
     * @param signStatus
     */
    @Async
    public void pushSignedOrder(String doctorId, String phone, String signId, String packageOrderId, Integer signStatus) {
        // 推送签名信息
        SyncSignedOrderDTO syncSignedOrderDTO = new SyncSignedOrderDTO();
        syncSignedOrderDTO.setDoctorId(String.valueOf(doctorId));
        syncSignedOrderDTO.setPhone(phone);
        syncSignedOrderDTO.setBenefitOrderId(packageOrderId);
        syncSignedOrderDTO.setSignSeqId(signId);
        syncSignedOrderDTO.setSignType(SignTypeEnum.SIGNED_DOCTOR.getValue());
        switch (signStatus) {
            case 2:
                syncSignedOrder(syncSignedOrderDTO);
                break;
            case 3:
                updateSignedOrder(syncSignedOrderDTO);
                break;
        }
    }

    @Async
    public void syncBenefitUsedOrder(NotifyConsultDataDTO notifyConsultDataDTO) {
        try {
            String param = JSONObject.toJSONString(notifyConsultDataDTO);
            log.info("syncBenefitUsedOrder请求入参{}", param);
            String result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.PUSH_DATA_URL, param, getThirdHead(notifyConsultDataDTO));
            log.info("syncBenefitUsedOrder请求返参{}", param);
        } catch (Exception e) {
            log.error("syncSignedOrder推送失败：{}", e);
        }
    }

    /**
     * 签约信息同步
     *
     * @param syncSignedOrderDTO
     */
    public void syncSignedOrder(SyncSignedOrderDTO syncSignedOrderDTO) {
        try {
            String param = JSONObject.toJSONString(syncSignedOrderDTO);
            log.info("syncSignedOrder请求入参{}", param);
            String result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.SYNC_SIGNED_ORDER_URL, param, getThirdHead(syncSignedOrderDTO));
            log.info("syncSignedOrder请求返参{}", param);
        } catch (Exception e) {
            log.error("syncSignedOrder推送失败：{}", e);
        }
    }

    /**
     * 改签约信息同步
     *
     * @param syncSignedOrderDTO
     */
    public void updateSignedOrder(SyncSignedOrderDTO syncSignedOrderDTO) {
        try {
            String param = JSONObject.toJSONString(syncSignedOrderDTO);
            log.info("updateSignedOrder请求入参{}", param);
            String result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.UPDATE_SIGNED_ORDER_URL, param, getThirdHead(syncSignedOrderDTO));
            log.info("updateSignedOrder请求返参{}", param);
        } catch (Exception e) {
            log.error("updateSignedOrder推送失败：{}", e);
        }

    }

    /**
     * 获取异常数据
     *
     * @param abnormalDataDTO
     */
    public void getAbnormalData(AbnormalDataDTO abnormalDataDTO) {
        try {
            String param = JSONObject.toJSONString(abnormalDataDTO);
            log.info("getAbnormalData请求入参{}", param);
            String result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.ABNORMAL_DATA_URL, param, getThirdHead(abnormalDataDTO));
            log.info("getAbnormalData请求返参{}", param);
        } catch (Exception e) {
            log.error("getAbnormalData请求失败：{}", e);
        }

    }

    /**
     * 查询用户健康计划是否开启状态
     *
     * @param healthInfoDTO
     */
    public void getHealthSchStatus(HealthInfoDTO healthInfoDTO) {
        try {
            String param = JSONObject.toJSONString(healthInfoDTO);
            log.info("getHealthSchStatus请求入参{}", param);
            String result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.HEALTH_SCHEDULE_STATUS_URL, param, getThirdHead(healthInfoDTO));
            log.info("getHealthSchStatus请求返参{}", param);
        } catch (Exception e) {
            log.error("getHealthSchStatus请求失败：{}", e);
        }

    }

    /**
     * 查询用户健康档案30天内是否有更新
     *
     * @param healthInfoDTO
     */
    public String getHealthRecordWhetherUpdate(HealthInfoDTO healthInfoDTO) {
        String result = "";

        try {
            String param = JSONObject.toJSONString(healthInfoDTO);
            log.info("getHealthRecordWhetherUpdate请求入参{}", param);
            result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.HEALTH_RECORD_WHETHER_UPDATE_URL, param, getThirdHead(healthInfoDTO));
            log.info("getHealthRecordWhetherUpdate请求返参{}", param);
        } catch (Exception e) {
            log.error("getHealthRecordWhetherUpdate请求失败：{}", e);
        }
        return result;
    }

    /**
     * 推送创建家庭医生图文随访订单
     *
     * @param followUpOrderDTO
     */
    public void pushFollowUpOrder(FollowUpOrderDTO followUpOrderDTO) {
        try {
            String param = JSONObject.toJSONString(followUpOrderDTO);
            log.info("pushFollowUpOrder请求入参{}", param);
            String result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.PUSH_FOLLOW_UP_ORDER_URL, param, getThirdHead(followUpOrderDTO));
            log.info("pushFollowUpOrder请求返参{}", param);
        } catch (Exception e) {
            log.error("pushFollowUpOrder请求失败：{}", e);
        }
    }

    /**
     * 查询用户头像
     *
     * @param userInfoDTO
     */
    public void getUserPic(UserInfoDTO userInfoDTO) {
        try {
            String param = JSONObject.toJSONString(userInfoDTO);
            log.info("getUserPic请求入参{}", param);
            String result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.USER_PIC_URL, param, getThirdHead(userInfoDTO));
            log.info("getUserPic请求返参{}", param);
        } catch (Exception e) {
            log.error("getUserPic请求失败：{}", e);
        }

    }

    /**
     * 获取用户基础信息
     *
     * @param userInfoDTO
     */
    public void getUserBasicInfo(UserInfoDTO userInfoDTO) {
        try {
            String param = JSONObject.toJSONString(userInfoDTO);
            log.info("getUserBasicInfo请求入参{}", param);
            String result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.USER_BASIC_INFO_URL, param, getThirdHead(userInfoDTO));
            log.info("getUserBasicInfo请求返参{}", param);
        } catch (Exception e) {
            log.error("getUserBasicInfo请求失败：{}", e);
        }

    }

    /**
     * 获取用户基础指标信息
     *
     * @param userInfoDTO
     */
    public void getUserBasicInd(UserInfoDTO userInfoDTO) {
        try {
            String param = JSONObject.toJSONString(userInfoDTO);
            log.info("getUserBasicInd请求入参{}", param);
            String result = HttpKit.jsonPost(projProperties.getFamilyDoctorThirdUrl() + UrlConstants.USER_BASIC_IND_URL, param, getThirdHead(userInfoDTO));
            log.info("getUserBasicInd请求返参{}", param);
        } catch (Exception e) {
            log.error("getUserBasicInd请求失败：{}", e);
        }
    }

}
